<!DOCTYPE html>
<html lang="en" class="dark">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width,initial-scale=1" />
  <title>Meet - Video Call (Pusher signaling)</title>
  <script src="../tailwind.js"></script>
  <!-- Pusher JS -->
  <script src="https://js.pusher.com/8.2.0/pusher.min.js"></script>

  <style>
    /* small animation you had */
    @keyframes fadeIn { from { opacity: 0; transform: translateY(20px);} to { opacity: 1; transform: translateY(0);} }
    .animate-fadeIn { animation: fadeIn 0.6s ease-out forwards; }
  </style>
</head>
<body class="bg-white dark:bg-gray-900 min-h-screen transition-colors duration-300">
  <div class="max-w-7xl mx-auto p-6 animate-fadeIn">
    <div class="flex items-center mb-8">
      <svg class="w-8 h-8 text-black dark:text-white" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
        <path stroke-linecap="round" stroke-linejoin="round" d="M3 8v8a3 3 0 003 3h12a3 3 0 003-3V8a3 3 0 00-3-3H6a3 3 0 00-3 3zm9 4a2 2 0 110-4 2 2 0 010 4z" />
      </svg>
      <span class="ml-2 text-2xl font-extrabold tracking-tight text-black dark:text-white">Meet</span>
    </div>

    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
      <div class="relative">
        <video id="local-video" class="w-full rounded-lg" autoplay muted playsinline></video>
        <p class="absolute bottom-2 left-2 text-white bg-black/50 px-2 py-1 rounded">You</p>
      </div>
      <div class="relative">
        <video id="remote-video" class="w-full rounded-lg" autoplay playsinline></video>
        <p class="absolute bottom-2 left-2 text-white bg-black/50 px-2 py-1 rounded">Remote</p>
      </div>
    </div>

    <div class="mt-4 flex space-x-4">
      <button id="mute-audio-btn" class="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600">Mute Audio</button>
      <button id="mute-video-btn" class="px-4 py-2 bg-gray-500 text-white rounded-lg hover:bg-gray-600">Mute Video</button>
      <button id="stop-btn" class="px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600">End Call</button>
    </div>

    <p id="call-error" class="mt-4 text-red-500 text-sm hidden"></p>
  </div>

  <!-- theme toggle omitted for brevity -->

<script>
/*
  Pusher-based WebRTC signaling client
  - Uses a presence channel "presence-call-<roomId>" so we can see how many users are in the room.
  - The FIRST user to join will create the offer (caller). Later joiners will respond with answer.
  - Signaling events: "offer", "answer", "candidate"
  - Uses server endpoint /server/send-signal.php to forward events through Pusher.
*/

const errorDisplay = document.getElementById('call-error');
const roomId = new URLSearchParams(window.location.search).get('room');
if (!roomId) {
  errorDisplay.textContent = 'No room ID provided';
  errorDisplay.classList.remove('hidden');
  throw new Error('No room ID provided');
}

// Replace with your Pusher key + cluster
const PUSHER_KEY = 'PUSHER_APP_KEY';
const PUSHER_CLUSTER = 'PUSHER_APP_CLUSTER';

// Initialize Pusher client (using presence channels requires an authEndpoint on server)
Pusher.logToConsole = false;
const pusher = new Pusher(PUSHER_KEY, {
  cluster: PUSHER_CLUSTER,
  authEndpoint: '/server/pusher_auth.php' // adjust path if needed
});

// presence channel name
const channelName = `presence-call-${roomId}`;
const channel = pusher.subscribe(channelName);

let peerConnection;
let localStream;
let userId; // will be filled by server via presence auth (Pusher includes user_id in subscription events)
let amICaller = false; // decide who creates offer

// simple helper to send signaling data to server which triggers Pusher
async function sendSignal(eventName, payload) {
  try {
    await fetch('/server/send-signal.php', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        roomId,
        event: eventName,
        payload
      }),
      credentials: 'include' // send cookies/session to auth
    });
  } catch (err) {
    console.error('sendSignal error', err);
  }
}

// Create RTCPeerConnection with STUN (add TURN in prod)
function createPeerConnection() {
  const config = {
    iceServers: [
      { urls: 'stun:stun.l.google.com:19302' }
      // add TURN here if you have one
    ]
  };
  peerConnection = new RTCPeerConnection(config);

  peerConnection.onicecandidate = (e) => {
    if (!e.candidate) return;
    // send candidate as plain object (toJSON())
    sendSignal('candidate', { user_id: userId, candidate: e.candidate.toJSON() });
  };

  peerConnection.ontrack = (e) => {
    const remoteVideo = document.getElementById('remote-video');
    remoteVideo.srcObject = e.streams[0];
  };

  peerConnection.oniceconnectionstatechange = () => {
    console.log('ICE state:', peerConnection.iceConnectionState);
    if (['disconnected', 'failed', 'closed'].includes(peerConnection.iceConnectionState)) {
      errorDisplay.textContent = 'Call disconnected';
      errorDisplay.classList.remove('hidden');
    }
  };

  return peerConnection;
}

// getUserMedia and attach local tracks
async function startLocalStream() {
  try {
    const stream = await navigator.mediaDevices.getUserMedia({ video: true, audio: true });
    localStream = stream;
    document.getElementById('local-video').srcObject = stream;
    // add tracks to PC
    stream.getTracks().forEach(track => peerConnection.addTrack(track, stream));
  } catch (err) {
    console.error('getUserMedia error', err);
    errorDisplay.textContent = 'Failed to access camera/microphone. Check permissions.';
    errorDisplay.classList.remove('hidden');
  }
}

// UI buttons
document.getElementById('mute-audio-btn').addEventListener('click', () => {
  if (!localStream) return;
  localStream.getAudioTracks().forEach(t => t.enabled = !t.enabled);
});
document.getElementById('mute-video-btn').addEventListener('click', () => {
  if (!localStream) return;
  localStream.getVideoTracks().forEach(t => t.enabled = !t.enabled);
});
document.getElementById('stop-btn').addEventListener('click', async () => {
  if (peerConnection) peerConnection.close();
  if (localStream) localStream.getTracks().forEach(t => t.stop());
  // optionally tell server to clear room data
  await fetch(`/server/send-signal.php`, {
    method: 'POST',
    headers: {'Content-Type':'application/json'},
    body: JSON.stringify({ roomId, event: 'hangup', payload: { user_id: userId } }),
    credentials: 'include'
  });
  window.location.href = 'dashboard.php';
});

// Pusher presence subscription success — gives member list
channel.bind('pusher:subscription_succeeded', (members) => {
  // members is a Pusher Members object
  // user_id for this client is available from pusher client? we get it through server auth response in pusher_auth.php
  console.log('subscribed to', channelName, 'members count =', members.count);
  // if only me in room -> I become caller
  if (members.count === 1) {
    amICaller = true;
    startCallAsCaller();
  } else {
    amICaller = false;
    // just wait for incoming offer events
  }
});

// Retrieving user_id from subscription_data: pusher fires member_added for others but not for self.
// We will rely on server-side to have set the user id in the auth response, but we can also request it via endpoint if needed.

// Bind signaling events
channel.bind('offer', async (data) => {
  console.log('offer received', data);
  if (!data || data.user_id === userId) return;
  // create PC if not exists
  if (!peerConnection) {
    createPeerConnection();
  }
  try {
    await peerConnection.setRemoteDescription(new RTCSessionDescription(data.offer));
    // add local stream if not already
    if (!localStream) await startLocalStream();
    // create answer
    const answer = await peerConnection.createAnswer();
    await peerConnection.setLocalDescription(answer);
    // send answer back through Pusher
    sendSignal('answer', { user_id: userId, answer: peerConnection.localDescription });
  } catch (err) {
    console.error('Error handling offer', err);
  }
});

channel.bind('answer', async (data) => {
  console.log('answer received', data);
  if (!data || data.user_id === userId) return;
  try {
    await peerConnection.setRemoteDescription(new RTCSessionDescription(data.answer));
  } catch (err) {
    console.error('Error setting remote answer', err);
  }
});

channel.bind('candidate', async (data) => {
  // candidate is plain object
  if (!data || data.user_id === userId) return;
  try {
    await peerConnection.addIceCandidate(new RTCIceCandidate(data.candidate));
  } catch (err) {
    console.error('Error adding ICE candidate', err);
  }
});

channel.bind('hangup', (data) => {
  console.log('hangup', data);
  // handle hangup (clean UI)
  if (peerConnection) peerConnection.close();
  peerConnection = null;
});

// If auth fails, Pusher will emit error messages in console. Listen to pusher:subscription_error if needed.
channel.bind('pusher:subscription_error', (status) => {
  console.error('Pusher subscription error', status);
  errorDisplay.textContent = 'Realtime auth failed. Check pusher_auth.php and sessions.';
  errorDisplay.classList.remove('hidden');
});

// Start call as caller: make PC, get media, create offer, send offer
async function startCallAsCaller() {
  console.log('starting call as caller...');
  peerConnection = createPeerConnection();
  await startLocalStream();
  try {
    const offer = await peerConnection.createOffer();
    await peerConnection.setLocalDescription(offer);
    // send offer via Pusher (server will broadcast back to other members)
    sendSignal('offer', { user_id: userId, offer: peerConnection.localDescription });
  } catch (err) {
    console.error('Error creating offer', err);
  }
}

// We need to obtain our own userId from the server (session). We'll fetch a small endpoint
// that returns the logged-in user's ID (server should return JSON { success:true, user_id: 'abc' }).
// Adjust path to your session check endpoint.
(async function fetchUserId() {
  try {
    const resp = await fetch('/server/auth/check_session.php', { credentials: 'include' });
    const data = await resp.json();
    if (!data.success) {
      window.location.href = 'login.php?error=Please log in';
      return;
    }
    userId = String(data.user.id); // ensure string
    console.log('my userId:', userId);
    // Note: presence auth on server should have already used session to register this user_id with Pusher.
    // The subscription already started above; we wait for `subscription_succeeded` to make the offer.
  } catch (err) {
    console.error('fetchUserId error', err);
    errorDisplay.textContent = 'Session error. Please log in again.';
    errorDisplay.classList.remove('hidden');
  }
})();
</script>
</body>
</html>