<?php
session_start();
require_once '../db/db_connect.php';

// Headers for JSON response and CORS
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);
$email = trim($input['email'] ?? '');
$password = trim($input['password'] ?? '');

try {
    // Validate input
    if (empty($email) || empty($password)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Email and password are required']);
        exit;
    }
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid email format']);
        exit;
    }

    // Check if user exists and is verified
    $stmt = $conn->prepare('SELECT id, email, password, username, is_verified FROM users WHERE email = ?');
    if (!$stmt) {
        throw new Exception('Failed to prepare statement: ' . $conn->error);
    }
    $stmt->bind_param('s', $email);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();

    if (!$user) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
        exit;
    }

    // Check if email is verified
    if ($user['is_verified'] == 0) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Please verify your email before logging in']);
        exit;
    }

    // Verify password
    if (!password_verify($password, $user['password'])) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Invalid credentials']);
        exit;
    }

// Set session variables
$_SESSION['user_id'] = $user['id'];
$_SESSION['email'] = $user['email'];
$_SESSION['username'] = $user['username'];

// Get redirect target if any
$redirectUrl = $_SESSION['redirect_after_login'] ?? null;
if ($redirectUrl) {
    unset($_SESSION['redirect_after_login']); // clear it after use
}

// Return success response
echo json_encode([
    'success' => true,
    'message' => 'Login successful',
    'redirect' => $redirectUrl ?: './dashboard', // fallback if none
    'user' => [
        'id' => $user['id'],
        'email' => $user['email'],
        'username' => $user['username']
    ]
]);
} catch (Exception $e) {
    error_log('Error in login.php: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['success' => false, 'message' => 'Server error. Please try again later.']);
} finally {
    if (isset($stmt)) {
        $stmt->close();
    }
    $conn->close();
}
?>